'use strict';

/* --------------------------------------------------------------
 content_header.js 2016-04-27
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Content Header Controller
 *
 * This module handles the behavior of the header controller. It will roll-in whenever the user scrolls to top.
 * The widget will emmit an event "content_header:roll_in" when it is rolled in and a "content_header:roll_out"
 * whenever it's hiding. These events can be useful if there's a need to re-position elements that are static
 * e.g. fixed table headers.
 *
 * In extend the content-header element will have the "fixed" class as long as it stays fixed on the viewport.
 */
gx.controllers.module('content_header', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * The original position of the tab navigation
     *
     * @type {Number}
     */
    var originalPosition = 0;

    /**
     * The last scroll position
     *
     * @type {Number}
     */
    var scrollPosition = $(window).scrollTop();

    /**
     * The original left position of the pageHeading
     *
     * @type {Number}
     */
    var originalLeft = 0;

    /**
     * Tells if the tab navigation is within the view port
     *
     * @type {Boolean}
     */
    var isOut = true;

    /**
     * Whether the content header is currently on animation.
     *
     * @type {Boolean}
     */
    var onAnimation = false;

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * On Window Scroll Handler
     *
     * Reset the page navigation frame to the original position if the user scrolls directly to top.
     */
    function _onWindowScroll() {
        if (onAnimation) {
            return;
        }

        var newScrollPosition = $(window).scrollTop();
        var isScrollDown = scrollPosition - newScrollPosition < 0;
        var isScrollUp = scrollPosition - newScrollPosition > 0;

        originalPosition = $('#main-header').height();
        originalLeft = $('#main-menu').outerWidth();

        _setContentHeaderAbsoluteLeft();

        var scrolledToTop = _checkScrolledToTop();

        if (!scrolledToTop) {
            if (isScrollDown && !isScrollUp && !isOut) {
                _rollOut();
            } else if (!isScrollDown && isScrollUp && isOut) {
                _rollIn();
            }
        }

        scrollPosition = newScrollPosition;
    }

    /**
     * Roll-in Animation Function
     */
    function _rollIn() {
        isOut = false;
        onAnimation = true;

        $this.trigger('content_header:roll_in');

        $this.css({
            top: '0',
            position: 'fixed'
        });

        // Retain the page height with a temporary padding.
        $this.parent().css('padding-top', $this.outerHeight() + 'px');

        $this.animate({
            top: originalPosition + 'px'
        }, {
            complete: function complete() {
                _checkScrolledToTop();
                onAnimation = false;
                _onWindowScroll(); // Check if it's necessary to re-render the position of the content-header.
                $this.addClass('fixed');
            }
        }, 'fast');
    }

    /**
     * Roll-out Animation Function
     */
    function _rollOut() {
        isOut = true;
        onAnimation = true;

        $this.trigger('content_header:roll_out');

        $this.animate({
            top: '0'
        }, 'fast', 'swing', function () {
            $this.css({
                top: originalPosition + 'px',
                position: ''
            });

            $this.parent().css('padding-top', ''); // Remove temporary padding.

            onAnimation = false;

            $this.removeClass('fixed');
        });
    }

    /**
     * Sets the left position of the pageHeading absolute
     */
    function _setContentHeaderAbsoluteLeft() {
        $this.css('left', originalLeft - $(window).scrollLeft());
    }

    /**
     * Check if user has scrolled to top of the page.
     *
     * @return {Boolean} Returns the check result.
     */
    function _checkScrolledToTop() {
        if ($(window).scrollTop() === 0) {
            $this.css({
                top: originalPosition + 'px',
                position: ''
            });

            $this.parent().css('padding-top', ''); // Remove temporary padding.

            return true;
        }

        return false;
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $(window).on('scroll', _onWindowScroll);
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
